%% Process MGGA, meteorological & met data
clear all;
close all;
%% Part 1: Generate Event Tables

                MGGA = readtable("\\csunas01.colostate.edu\research\METEC\User\mbua\ADED\CH4 MGGA\CH4 MGGA.csv");
                        
                    MGGA.Time = datetime(MGGA.Time, 'InputFormat', 'MM/dd/yyyy HH:mm:ss.SSS'); % Convert to datetime
                    MGGA.Time = dateshift(MGGA.Time, 'end', 'second'); % Round to the nearest second
                    MGGA.Time.Format = 'MM/dd/yyyy HH:mm:ss'; % Set the desired format
                        % Filter MGGA data to have till March 20 2024 when
                        % MGGA broke down
                     
                % Define the end time of 03/20/2024
                end_time = datetime(2024, 3, 21, 0, 0, 0); % End of March 20, 2024
                
                % Filter the data to include only entries up to the end time of 03/20/2024
                MGGA = MGGA(MGGA.Time <= end_time, :);


                      

                met = readtable("\\csunas01.colostate.edu\research\METEC\User\mbua\ADED\MET2_DATA\Processed Met\met.csv");
                    met.Time = datetime(met.Time, 'InputFormat', 'MM/dd/yyyy HH:mm:ss.SSS'); % Convert to datetime
                    met.Time = dateshift(met.Time, 'end', 'second'); % Round to the nearest second
                    met.Time.Format = 'MM/dd/yyyy HH:mm:ss'; % Set the desired format

                
                % Step 2: Filter condition
                filter_time = datetime(2024, 2, 8, 13, 22, 22);
                
                filter_condition_MGGA = MGGA.Time >= filter_time;
                filter_condition_met = met.Time >= filter_time;
                
                % Apply the filter to the tables
                MGGA_filtered = MGGA(filter_condition_MGGA, :);
                met_filtered = met(filter_condition_met, :);
                
                % Step 4: Calculate second averages by grouping rows with the same timestamp
                MGGA_second_avg = varfun(@mean, MGGA_filtered, 'GroupingVariables', 'Time', 'OutputFormat', 'table');
               met_second_avg = varfun(@mean, met_filtered, 'GroupingVariables', 'Time', 'OutputFormat', 'table');
                
                    MGGA_second_avg = removevars(MGGA_second_avg, 'GroupCount');
                   met_second_avg = removevars(met_second_avg, 'GroupCount');
                
                
                % Merge CH4 data to MET data
                
                MGGA_met = outerjoin(met_second_avg, MGGA_second_avg, 'Keys', 'Time');
                    MGGA_met = rmmissing(MGGA_met);
                    MGGA_met = renamevars(MGGA_met, 'Time_met_second_avg','Time');
disp('Part 1 Complete')



%% % Part 2: Add release data to the files

% Step 1: Add MGGA & Aeris locations
MGGA_lat = repmat(40.59603590, height(MGGA_met), 1); 
MGGA_long = repmat(-105.14032186, height(MGGA_met), 1);
MGGA_met = addvars(MGGA_met, MGGA_lat, MGGA_long, 'NewVariableNames', {'MGGA_lat_N', 'MGGA_long_W'});

disp('Part 2 Complete')
%% Part 3: Add METEC release data
% Read releases table
releases = readtable('\\csunas01.colostate.edu\research\METEC\User\mbua\ADED\Releases\Releases_Information.csv');

% Convert UTC start and end times to Mountain Time (America/Denver)
utcTimeZone = 'UTC';
mtTimeZone = 'America/Denver';

% Convert UTC start and end times to datetime objects
releases.UTCStart = datetime(releases.UTCStart, 'InputFormat', 'yyyy-MM-dd HH:mm:ss', 'TimeZone', utcTimeZone);
releases.UTCEnd = datetime(releases.UTCEnd, 'InputFormat', 'yyyy-MM-dd HH:mm:ss', 'TimeZone', utcTimeZone);

% Convert UTC start and end times to Mountain Time
releases.MTStart = datetime(releases.UTCStart, 'TimeZone', mtTimeZone);
releases.MTEnd = datetime(releases.UTCEnd, 'TimeZone', mtTimeZone);

% Initialize a new empty table to store the merged data for MGGA_met
MGGA_met_releases = table();

% Convert MGGA_met.Time to America/Denver time zone
MGGA_met.Time = datetime(MGGA_met.Time, 'TimeZone', 'America/Denver');

% Iterate over each row in releases                        
for i = 1:height(releases)
    % Extract the MTStart and MTEnd from releases for the current row
    MTStart = releases.MTStart(i);
    MTEnd = releases.MTEnd(i);
    
    % Find rows in MGGA_met where the Time falls within the MTStart and MTEnd interval
    mask = MGGA_met.Time >= MTStart & MGGA_met.Time <= MTEnd;
    metInRange = MGGA_met(mask, :);
    
    % If there are rows in metInRange, add the release information from releases
    if ~isempty(metInRange)
        % Add the release information from releases to metInRange
        % by adding new columns for each variable in releases
        releaseData = repmat(table2struct(releases(i, :)), height(metInRange), 1);
        releaseData = struct2table(releaseData, 'AsArray', true);
        metInRange = [metInRange, releaseData];
        
        % Append the merged data to MGGA_met_releases
        MGGA_met_releases = [MGGA_met_releases; metInRange];
    end
end


disp('Part 3 Complete')
%% %% Part 4: Split ch4_met_releases_tables into event tables 

% for MGGA
% Convert datetime values to serial date numbers
MTStart_serial = datenum(MGGA_met_releases.MTStart);
MTEnd_serial = datenum(MGGA_met_releases.MTEnd);

% Round off the serial date numbers to the nearest minute
rounded_MTStart = round(MTStart_serial * 24 * 60) / (24 * 60);
rounded_MTEnd = round(MTEnd_serial * 24 * 60) / (24 * 60);

% Find unique combinations of rounded MTStart and MTEnd
unique_combinations = unique([rounded_MTStart, rounded_MTEnd], 'rows');

% Initialize a cell array to store mini tables
mini_tables = cell(size(unique_combinations, 1), 1);

% Loop through each unique combination and create mini tables for MGGA
for i = 1:size(unique_combinations, 1)
    % Find rows that match the current combination
    matching_rows = all([rounded_MTStart, rounded_MTEnd] == unique_combinations(i, :), 2);
    
    % Create a mini table for the current combination for MGGA
    mini_tables{i} = MGGA_met_releases(matching_rows, :);
end

% Define the directory where the event tables will be saved for MGGA
save_directory_MGGA = '\\csunas01.colostate.edu\research\METEC\User\mbua\ADED\OTM33_Gaussian\Event Tables\MGGA';

% Check if the directory exists, if not, create it
if ~exist(save_directory_MGGA, 'dir')
    mkdir(save_directory_MGGA);
end

% Loop through each mini table for MGGA and save it as an event table
for i = 1:numel(mini_tables)
    % Get the start and end times for the current mini table for MGGA
    MTStart = mini_tables{i}.MTStart(1);
    MTEnd = mini_tables{i}.MTEnd(1);
    
    % Create the filename for the event table for MGGA
    filename = sprintf('%s_%s MGGA Event table.csv', datestr(MTStart, 'yyyymmddHHMM'), datestr(MTEnd, 'yyyymmddHHMM'));
    
    % Save the mini table as a CSV file in the specified directory for MGGA
    writetable(mini_tables{i}, fullfile(save_directory_MGGA, filename));
end

disp('Part 4 Complete')
%% %% Part 5: Add distance to the tables
% Define the directory path
directory_path = "\\csunas01.colostate.edu\research\METEC\User\mbua\ADED\OTM33_Gaussian\Event Tables";

% Get all items in the directory
folder_list = dir(directory_path);

% Keep only directories that contain "MGGA" in the name
folder_list = folder_list([folder_list.isdir]);  % Keep only folders
folder_list = folder_list(contains({folder_list.name}, "MGGA"));  % Filter by name

% Loop through each folder
for i = 1:numel(folder_list)
    % Skip if it's a special folder
    if strcmp(folder_list(i).name, '.') || strcmp(folder_list(i).name, '..')
        continue;
    end
    
    % Check if it's a directory
    if folder_list(i).isdir
        % Get the folder name
        folder_name = folder_list(i).name;
        
        % Get the column names based on folder name
        if contains(folder_name, "MGGA")
            long_col_name = "MGGA_long_W";
            lat_col_name = "MGGA_lat_N";
        elseif contains(folder_name, "Aeris_225")
            long_col_name = "Aeris_long_W";
            lat_col_name = "Aeris_lat_N";
        elseif contains(folder_name, "Aeris_259")
            long_col_name = "Aeris_long_W";
            lat_col_name = "Aeris_lat_N";
        else
            % Skip processing if folder name doesn't match expected patterns
            continue;
        end
        
        % Get a list of all CSV files in the folder
        file_list = dir(fullfile(directory_path, folder_name, '*.csv'));

        % Loop through each CSV file in the folder
        for j = 1:numel(file_list)
            % Read the table from the CSV file
            filename = fullfile(directory_path, folder_name, file_list(j).name);
            file = readtable(filename);

            % Calculate distances
            distance_meters = haversine(file.Lat, file.Lon, file.(lat_col_name), file.(long_col_name));

            % Add distances to the file table
            file.Distance_meters = distance_meters;

            % Write the modified table back to the original CSV file
            writetable(file, filename);
        end
    end
end
% Define the haversine function
function distance = haversine(lat1, lon1, lat2, lon2)
    % Earth radius in meters
    R = 6371000;

    % Convert degrees to radians
    lat1 = deg2rad(lat1);
    lon1 = deg2rad(lon1);
    lat2 = deg2rad(lat2);
    lon2 = deg2rad(lon2);

    % Calculate differences
    dlat = lat2 - lat1;
    dlon = lon2 - lon1;

    % Haversine formula
    a = sin(dlat/2).^2 + cos(lat1) .* cos(lat2) .* sin(dlon/2).^2;
    c = 2 * atan2(sqrt(a), sqrt(1-a));
    distance = R * c;
end


%% 


